"""
Copyright 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
"""

import signal
import sys
import os

from subprocess import Popen, PIPE
from textwrap import TextWrapper

from vmis.db import db
from vmis.core.common import State
from vmis.core.errors import ValidationError, ValidationErrorNonFatal, EULADeclinedError
from vmis.core.errors import CPUCheckError, IgnoreCPUCheckError
from vmis.core.questions import ValidationError, QUESTION_DIRECTORY, \
                                QUESTION_YESNO, QUESTION_NUMERIC, \
                                QUESTION_CLOSEPROGRAMS, QUESTION_TEXTENTRY, \
                                QUESTION_SHUTDOWNPROGRAM, QUESTION_PORTENTRY, \
                                QUESTION_DUALPORTENTRIES
from vmis.ui import MessageTypes
from vmis.ui.uiAppControl import UIAppControl
from vmis.util import Format
from vmis.util.log import getLog
from pathlib import Path
from vmis.util.shell import Which, run

log = getLog('vmis.ui.base')

class EULA(State):
   @staticmethod
   def Show(txn, eula):
      """
      Show the EULA to the user and get acceptance of it.
      """
      answer = eula.GetDefault()

      while True:
         answer = txn.ui.ShowEULA(eula.text, eula.componentName)
         try:
            eula.Validate(answer)
            txn.Next()
            break
         except EULADeclinedError:
            raise     # When the user explicitly declines we bail out.
         except ValidationError as e:
            print(Format(str(e)), file=sys.stderr)
            print()

class CPUCheck(State):
   @staticmethod
   def Show(txn, deprecatedCPU):
      text = 'The host CPU does not support the necessary hardware requirements.\n' + \
             'If you Ignore this message to proceed with installation, \n' + \
             'you may not be able to launch virtual machines on this host.\n' + \
             'For details, please refer to the following Knowledge Base article:\n' + \
             'https://kb.vmware.com/kb/51643'
      prompt = 'Do you still want to install this product? [yes/no]'

      print(text)
      while True:
         answer = txn.ui.Prompt(prompt, '', format=False)
         try:
            deprecatedCPU.Validate(answer)
            txn.Next()
            break
         except CPUCheckError:
            raise
         except IgnoreCPUCheckError as e:
            pass

def TextEntry(question, text, default, ui):
   """
   Populate a text entry

   @param question: The question object attached to this call
   @param text: Text to display. Not used in this function.
   @param default: Default entry.  Not used in this function
   @param ui: The user interface object.  Not used in this function

   @return: (Text with which to prompt the user, None)
   """
   text = "%s\n%s\n" % (question.header, question.footer)
   return (text, None)


def ShutdownProgram(question, text, default, ui):
   """
   Detect and shutdown a specific program

   @param question: The question object attached to this call
   @param text: Text to display. Not used in this function.
   @param default: Default entry.  Not used in this function
   @param ui: The user interface object

   @return: (None, None) - Move on to the next question when done
   """
   programShutdown = False
   while not programShutdown:
      ret = run('/bin/sh', '-c', '/bin/ps -e | grep %s' % question.program, ignoreErrors=True)['stdout']

      if ret:
         # Found some processes that match the criteria.  Prompt the user and stay
         # in the loop
         text = 'The VMware Installer cannot continue while %s ' % question.programName + \
          'is running.  Please shut it down and press <Enter> to continue.'
         answer = ui.Prompt(text, None, format=False)
      else:
         # Otherwise we're good.
         programShutdown = True
   return (None, None)


def ClosePrograms(question, text, default, ui):
   """
   Display the close programs and VMs question.

   @param question: The question object attached to this call
   @param text: Text to display
   @param default: Default entry
   @param ui: The user interface object

   @return: (None, None) - Move on to the next question when done
   """
   # Loop until there are no open items
   appControlSucceeded = False
   while True:
      try:
         text = ""

         ui.appControl.Initialize()

         openList = []

         # If there are open items, retrieve their names.
         if (ui.appControl.numVMs > 0):
            for i in range(ui.appControl.numVMs):
               name = ui.appControl.GetVMInfo(i)
               openList.append(name)

         if (ui.appControl.numApps > 0):
            for i in range(ui.appControl.numApps):
               (name, product) = ui.appControl.GetAppInfo(i)
               openList.append(name)

         if openList:
            text = 'The following virtual machines and VMware applications\n' + \
                   'are running.  Please suspend or close them, and then press\n' +\
                   '\'Enter\' key to continue. Or directly press \'Enter\' key to\n' + \
                   'do this automatically. You can also press any other key,\n' + \
                   'and then press \'Enter\' key to cancel the installation.\n\n'
            for item in openList:
               text = text + '> %s\n' % item

            answer = ui.Prompt(text, None, format=False)

            if answer != None:
               return (None, 'Cancel')
            else:
               ui.appControl.ShutdownAll()
               appControlSucceeded = True
         else:
            return (None, None)
      except Exception as e:
         log.info('Cannot use vmware-app-control to shut down open VMs, defaulting to fallback message.')
         log.debug('Exception: %s' % e)

      # Fall back on our old method.  It can't detect running UIs, only running VMs, but it's
      # better than nothing.  Only allow installation to continue when no more VMs are running.
      if question.checkVMsRunning():
         if appControlSucceeded:
            text = 'The VMware Installer could not shut down all running virtual ' + \
                   'machines.  If you have encrypted VMs open, please shut them down or ' + \
                   'suspend them now and press \'enter\' to continue.'
         else:
            text = 'The VMware Installer cannot continue if there are running virtual\n' + \
                   'machines. Shut down or suspend running virtual machines before\n' + \
                   'continuing.  Press \'enter\' to continue.'
         answer = ui.Prompt(text, None, format=False)
      else:
         return (None, None)

def PortEntry(question, text, default, ui):
   """
   Grab a port entry

   @param question: The question object attached to this call
   @param text: Text to display. Not used in this function.
   @param default: Default entry. Not used in this function
   @param ui: The user interface object

   @return: (None, Answer)
   """
   while 1:
      port = ''
      if default:
         port = default

      answer = ui.Prompt('%s (%s)' % (text, question.label), port, format=True)
      try:
         question.Validate(answer)
         return (None, answer)
      except ValidationErrorNonFatal as e:
         # An explicitly non-fatal error.
         return (None, answer)
      except ValidationError as e:
         print(e)

def DualPortEntries(question, text, default, ui):
   """
   Grab two port entries

   @param question: The question object attached to this call
   @param text: Text to display. Not used in this function.
   @param default: Default entry.  Not used in this function
   @param ui: The user interface object

   @return: (None, Answer)
   """
   portsValid = False
   while not portsValid:
      port1 = ''
      port2 = ''
      if default: # Parse it out
         ports = default.split('/')
         if len(ports) == 2:
            port1 = ports[0]
            port2 = ports[1]

      string = '%s (%s)' % (text, question.label1)
      answer1 = ui.Prompt(string, port1, format=True)
      string = '%s (%s)' % (text, question.label2)
      answer2 = ui.Prompt(string, port2, format=True)
      # This will throw an exception if the answers are invalid.
      try:
         answer = '%s/%s' % (answer1, answer2)
         question.Validate(answer)
         return (None, answer)
      except ValidationErrorNonFatal as e:
         # An explicitly non-fatal error.  Return our answer.
         return (None, answer)
      except ValidationError as e:
         print(e)
# Mapping between question types and their functions.
questionFunctions = { QUESTION_DIRECTORY : None,
                      QUESTION_YESNO : None,
                      QUESTION_NUMERIC : None,
                      QUESTION_CLOSEPROGRAMS: ClosePrograms,
                      QUESTION_SHUTDOWNPROGRAM: ShutdownProgram,
                      QUESTION_TEXTENTRY: TextEntry,
                      QUESTION_PORTENTRY: PortEntry,
                      QUESTION_DUALPORTENTRIES: DualPortEntries,}

class Question(State):
   @staticmethod
   def Show(txn, question):
      while True:
         # Allow for more elaborate prompts
         func = questionFunctions.get(question.type, None)
         answer = None
         if func != None:
            (text, answer) = func(question, question.text, question.GetDefault(), txn.ui)
         else:
            text = question.text
         # If an answer hasn't been set, check the text entry
         if not answer:
            if text is None:
               # If the returned text was None, move on to the next question.
               txn.Next()
               break
            else:
               answer = txn.ui.Prompt(text, question.GetDefault())
         try:
            answer = question.Validate(answer)
            db.config.Set(question.component, question.key, answer)
            txn.Next()
            break
         except ValidationErrorNonFatal as e:
            print(Format(str(e)), file=sys.stderr)
            # This is an explicitly non-fatal case.  Continue with the given answers.
            db.config.Set(question.component, question.key, answer)
            txn.Next()
            break
         except ValidationError as e:
            print(Format(str(e)), file=sys.stderr)

class Finish(State):
   @staticmethod
   def Show(txn, state):
      txn.ui.ShowFinish(txn.success, txn.message)
      txn.Quit()

class PromptInstall(State):
   @staticmethod
   def Show(txn, state):
      txn.ui.ShowPromptInstall()
      txn.Next()

def ShowMessage(messageType, message, useWrapper=True):
   """
   Write a message to the terminal

   If messageType is of greater or equal to severity of
   MessageType.WARNING the message is written to stderr.  Otherwise
   the message is written to stdout.

   @param messageType: one of MessageType
   @param message: message text to display
   @param useWrapper: Bool: Wrap message text or not
   """
   if messageType >= MessageTypes.WARNING:
      output = sys.stderr
   else:
      output = sys.stdout

   if useWrapper:
      output.write('%s\n' % Format(message))
   else:
      output.write('%s\n' % message)

class Wizard(object):
   """ Console UI wizard """
   HEADER = '['
   FOOTER = ']'
   PERCENT = '%4s%%'
   WIDTH = 70

   def __init__(self, txn):
      # Set up App Control.
      try:
         self.appControl = UIAppControl()
      except:
         self.appControl = None

   def ShowFinish(self, success, message):
      """ Clean up and restore console state """
      # Newline before printing our final message.
      print()

      if success:
         print(Format(message))
      else:
         print(Format(message), file=sys.stderr)

   def UserMessage(self, messageType, message, useWrapper=False):
      """ In console, a passthrough to ShowMessage """
      ShowMessage(messageType, message, useWrapper=useWrapper)

   def ShowPromptInstall(self):
      input(Format('The product is ready to be installed.  Press Enter to begin '
                       'installation or Ctrl-C to cancel.'))
      print()

   def ShowEULA(self, text, productName):
      """ Display a EULA and ask for acceptance """
      input(Format('You must accept the %s End User License'
                       ' Agreement to continue.  Press Enter to proceed.' % productName))

      # @fixme 1.0:  need some sane way to handle helpers like this
      try:
         # The EULA should be given without any wrapping already done
         # to it.  However, TextWrapper assumes that the text has not been
         # formatted already.  Since the EULA contains line breaks, this
         # throws off TextWrapper's formatting.  So we need to format it line
         # by line instead of in one large chunk.
         wrapper = TextWrapper()
         wrapper.width = 79
         wrapper.replace_whitespace = False # Needed to preserve paragraph spacing.
         EUList = text.split('\n')
         text = ""
         for line in EUList:
            text = ''.join((text, wrapper.fill(line), '\n'))

         Popen(self._getMoreBin(), stdin=PIPE).communicate(input=str.encode(text))
      except IOError as e: # RHEL4 appears to close stdin while we still expect it to be open
         pass

      return self.Prompt('Do you agree? [yes/no]', '')

   def _getMoreBin(self):
      """
      Respect user's $PAGER (or fallback to `more' if none is specified).
      If $PAGER is set to `less', add the `-E' flag to exit as soon as
      EOF is reached.

      @return: If the pager is found, a string or list of strings containing the
      command to execute the pager.  'more' if no alternative is found.
      """
      pager = os.environ.get('PAGER')

      if pager:
         # If the pager is not an absolute path, search the
         # path for it.
         if not Path(pager).is_absolute():
            pager = Which(pager)
         elif not os.access(Path(pager).as_posix(), os.X_OK):
            # Verify that the pager exists and is executable.
            # Don't just assume that it's been set correctly.
            # This keeps us from crashing.
            pager = None

      # If no pager is found, or cannot be found in
      # the path, default to 'more'.
      if not pager:
         pager = 'more'

      # Append -E to less so it quits after the last line of
      # the EULA is displayed.
      if Path(pager).name == 'less':
         pager = (pager, '-E')

      return pager

   def Prompt(self, text, default, format=True):
      """
      Prompt for an answer

      @fixme: This is rather awkward because we have hijacked SIGINT for
      our own _abort to prevent cancellation when we're in an
      inconsistent state.  However, we need a way to cancel from here.
      Right now we're doing that by letting the EOFError pass through.
      """
      defaultTxt = default and ' [%s]' % default or ''

      if format:
         answer = input('%s: ' % Format(text + defaultTxt))
      else:
         answer = input('%s: ' % (text + defaultTxt))
      print()

      if not answer:
         answer = default

      return answer

   def SetProgress(self, fraction):
      """ Print a progress bar at the given fraction """
      pass

   def EnableCursor(self, enable):
      """ Enable or disable the cursor """
      pass

   def _moveBeginOfLine(self):
      """ Move to the beginning of the current line """
      pass

   def SetPrimaryText(self, txt):
      pass

   def SetSecondaryText(self, txt):
      pass

   def EnableBack(self, enabled):
      pass

   def EnableNext(self, enabled):
      pass

   def HideCancel(self):
      pass

   def HideBack(self):
      pass

   def HideNext(self):
      pass

   def SetTitle(self, title):
      pass

   def EnableCancel(self, enable):
      """ Enable/disable cancellation by setting or ignoring SIGINT """
      # EnableCancel(True) might be called even if cancellation is
      # enabled so we must key off the previously saved state as well.
      #
      # XXX: use hasattr because null UI doesn't use console's
      # __init__.
      # If we've never set this variable, define it now to False.  We need
      # this so self._cancelFunction doesn't get overridden by two calls
      # with enable set to False.
      if not hasattr(self, '_cancelEnabled'):
         self._cancelEnabled = False

      if enable and not self._cancelEnabled and hasattr(self, '_cancelFunction'):
         self._cancelEnabled = True
         signal.signal(signal.SIGINT, self._cancelFunction)

      if not enable and self._cancelEnabled:
         self._cancelFunction = signal.getsignal(signal.SIGINT)
         signal.signal(signal.SIGINT, signal.SIG_IGN)
         self._cancelEnabled = False

   def SetNextType(self, type):
      pass

   def _moveLine(self, line):
      """ Move the cursor to given line number at the beginning of the line """
      pass

   def SetPrimaryProgressMessage(self, text):
      pass

   def SetSecondaryProgressMessage(self, text):
      pass

   def ShowProgress(self):
      """ Initialize progress display """
      pass
